unit xJoy;

(**************************************************************)
(** //              XENGINE Joystick Unit                 // **)
(** //  (C) 2025 Coded by Adam Kozinski & Dominik Galoch  // **)
(** //////////////////////////////////////////////////////// **)
(**************************************************************)

interface

const
    JOYSTICK_PORT = $201; { Joystick port definition }

var
    JOY_LEFT, JOY_RIGHT, JOY_UP, JOY_DOWN,                           { Joystick direction flags }
    JOY_BUTTON_1, JOY_BUTTON_2, JOY_BUTTON_PRESSED, JOY_WAITBUTTON,  { Button flags }
    JOY_ENABLED, JOY_CALIBRATED : boolean;                           { Joystick state flags }

type TJoystick = record
                    X, Y, XLeft, XRight, YUp, YDown,                 { Joystick position }
                    XCenter, YCenter, XMin, XMax, YMin, YMax : Word; { Calibration parameters }
                end;

var
    Joy : TJoystick; { Variable storing joystick state }

{////////////////////////////////////////////////////////////////}
{                  PROCEDURE AND FUNCTION HEADERS                }
{////////////////////////////////////////////////////////////////}

procedure xJoyInit;      { Joystick initialization }
procedure xJoyUpdate;    { Update joystick state }
procedure xJoyReset;     { Reset joystick }
procedure xJoyCalibrate; { Calibrate joystick }

{////////////////////////////////////////////////////////////////}

implementation

(**********************************************************)

procedure xJoyInit;
begin
    JOY_ENABLED := TRUE; { Set joystick enabled flag }
end;

(**********************************************************)

procedure xJoyUpdate;
begin
    Joy.X := 0; { Reset X position }
    Joy.Y := 0; { Reset Y position }

    if JOY_ENABLED then
    begin
        asm
            pushf
            xor cx, cx              { Clear CX register }
            mov dx, JOYSTICK_PORT   { Set joystick port }
            in  al, dx              { Read from port }
            not al
            mov ah, al
            and ax, 2010h           { Check buttons }
            mov JOY_BUTTON_1, al    { Store button 1 state }
            shr ah, 1
            mov JOY_BUTTON_2, ah    { Store button 2 state }
            mov bx, 0201h
            cli
            out dx, al              { Write to port }
            mov ah, 0
        @@1:
            in   al, dx             { Read joystick data }
            and  al, bl
            add  Joy.X, ax          { Accumulate X data }
            in   al, dx
            and  al, bh
            shr  al, 1
            add  Joy.Y, ax          { Accumulate Y data }
            in   al, dx
            test al, 3
            jz   @@2
            inc  cx
            jnz  @@1
            mov  JOY_ENABLED, 0     { Disable joystick if not responding }
        @@2:
            popf
        end;

        if JOY_ENABLED and (not JOY_WAITBUTTON) then
        begin
            { Check direction states based on X and Y values }
            JOY_LEFT  := (Joy.X < Joy.XLeft);
            JOY_RIGHT := (Joy.X > Joy.XRight);
            JOY_UP    := (Joy.Y < Joy.YUp);
            JOY_DOWN  := (Joy.Y > Joy.YDown);
        end
        else
        begin
            { Disable directions when not waiting for button }
            JOY_LEFT  := FALSE;
            JOY_RIGHT := FALSE;
            JOY_UP    := FALSE;
            JOY_DOWN  := FALSE;

            JOY_BUTTON_PRESSED := JOY_BUTTON_PRESSED or JOY_BUTTON_1 or JOY_BUTTON_2;
            JOY_BUTTON_1 := FALSE;
            JOY_BUTTON_2 := FALSE;
        end;
    end;
end;

(**********************************************************)

procedure xJoyReset;
var
    i : Integer;
    TotalX, TotalY : LongInt;
begin
    xJoyInit;   { Initialize joystick }
    xJoyUpdate; { Update joystick state }

    if JOY_ENABLED then
    begin
        TotalX := 0; { Reset X sum }
        TotalY := 0; { Reset Y sum }

        for i := 1 to 16 do
        begin
            xJoyUpdate;         { Update joystick state }
            Inc(TotalX, Joy.X); { Add X value to sum }
            Inc(TotalY, Joy.Y); { Add Y value to sum }
        end;

        { Calculate average X and Y values }
        Joy.X       := TotalX div 16;
        Joy.Y       := TotalY div 16;
        Joy.XCenter := Joy.X;
        Joy.YCenter := Joy.Y;
        Joy.XMin    := Joy.X;
        Joy.YMin    := Joy.Y;
        Joy.XMax    := Joy.X;
        Joy.YMax    := Joy.Y;
        Joy.XLeft   := Joy.X;
        Joy.YUp     := Joy.Y;
        Joy.XRight  := Joy.X;
        Joy.YDown   := Joy.Y;

        xJoyUpdate;
        JOY_CALIBRATED := FALSE; { Set calibration flag to false }
    end;
end;

(**********************************************************)

procedure xJoyCalibrate;
begin
    xJoyUpdate; { Update joystick state }

    if JOY_ENABLED then
    begin
        { Check X and Y values and update min/max values }
        if Joy.X < Joy.XMin then
        begin
            Joy.XMin  := Joy.X;
            Joy.XLeft := (Joy.XMin + Joy.XCenter) div 2;
        end;

        if Joy.Y < Joy.YMin then
        begin
            Joy.YMin := Joy.Y;
            Joy.YUp  := (Joy.YMin + Joy.YCenter) div 2;
        end;

        if Joy.X > Joy.XMax then
        begin
            Joy.XMax   := Joy.X;
            Joy.XRight := (Joy.XMax + Joy.XCenter) div 2;
        end;

        if Joy.Y > Joy.YMax then
        begin
            Joy.YMax  := Joy.Y;
            Joy.YDown := (Joy.YMax + Joy.YCenter) div 2;
        end;

        JOY_CALIBRATED := TRUE; { Set calibration flag to true }
    end;
end;

(**********************************************************)

BEGIN
    JOY_CALIBRATED := FALSE; { Set calibration flag to false }
    JOY_WAITBUTTON := FALSE; { Set wait-for-button flag to false }
    xJoyUpdate;              { Update joystick state }
END.
